<?php
/**
 * Plugin Name: AI Chat Widget
 * Plugin URI: https://example.com/ai-chat-widget
 * Description: نافذة دردشة ذكية للرد على العملاء باستخدام الذكاء الاصطناعي - تدعم OpenAI و DeepSeek
 * Version: 1.0.0
 * Author: Developer
 * Author URI: https://example.com
 * Text Domain: ai-chat-widget
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 8.0
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// منع الوصول المباشر
if (!defined('ABSPATH')) {
    exit;
}

// تعريف الثوابت
define('AICW_VERSION', '1.0.0');
define('AICW_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AICW_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AICW_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * الكلاس الرئيسي للإضافة
 */
final class AI_Chat_Widget {
    
    /**
     * Instance وحيد من الكلاس
     */
    private static $instance = null;
    
    /**
     * الحصول على Instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * تحميل الملفات المطلوبة
     */
    private function load_dependencies() {
        require_once AICW_PLUGIN_DIR . 'includes/class-ai-providers.php';
        require_once AICW_PLUGIN_DIR . 'includes/class-knowledge-base.php';
        require_once AICW_PLUGIN_DIR . 'includes/class-chat-handler.php';
        require_once AICW_PLUGIN_DIR . 'includes/class-admin-settings.php';
    }
    
    /**
     * تهيئة الـ Hooks
     */
    private function init_hooks() {
        // تفعيل الإضافة
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // إلغاء تفعيل الإضافة
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // تحميل ملفات الترجمة
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        
        // تحميل السكربتات والستايلات للواجهة الأمامية
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        
        // تحميل السكربتات والستايلات للوحة التحكم
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // إضافة نافذة الدردشة في الفوتر
        add_action('wp_footer', array($this, 'render_chat_widget'));
        
        // تسجيل AJAX handlers
        add_action('wp_ajax_aicw_send_message', array('AICW_Chat_Handler', 'handle_message'));
        add_action('wp_ajax_nopriv_aicw_send_message', array('AICW_Chat_Handler', 'handle_message'));
    }
    
    /**
     * عند تفعيل الإضافة
     */
    public function activate() {
        // إنشاء جدول قاعدة البيانات
        $this->create_database_table();
        
        // إضافة الإعدادات الافتراضية
        $this->set_default_options();
        
        // مسح الكاش
        flush_rewrite_rules();
    }
    
    /**
     * إنشاء جدول قاعدة البيانات
     */
    private function create_database_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aicw_chat_logs';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            session_id VARCHAR(100) NOT NULL,
            user_message TEXT NOT NULL,
            bot_response TEXT NOT NULL,
            tokens_used INT(11) DEFAULT 0,
            ip_address VARCHAR(45),
            user_agent TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY session_id (session_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * إضافة الإعدادات الافتراضية
     */
    private function set_default_options() {
        $default_options = array(
            'ai_provider' => 'openai',
            'openai_api_key' => '',
            'openai_model' => 'gpt-3.5-turbo',
            'deepseek_api_key' => '',
            'deepseek_model' => 'deepseek-chat',
            'temperature' => 0.7,
            'max_tokens' => 1000,
            'knowledge_base' => '',
            'system_prompt' => 'أنت مساعد ذكي ودود. ساعد العملاء بأفضل طريقة ممكنة. كن مهذباً ومختصراً في إجاباتك.',
            'primary_color' => '#0073aa',
            'secondary_color' => '#ffffff',
            'widget_position' => 'right',
            'welcome_message' => 'مرحباً! كيف يمكنني مساعدتك اليوم؟',
            'bot_name' => 'المساعد الذكي',
            'bot_avatar' => '',
            'enabled_pages' => 'all',
            'excluded_pages' => '',
            'rate_limit' => 10,
            'enable_sound' => true,
            'enable_logging' => true,
        );
        
        if (!get_option('aicw_settings')) {
            add_option('aicw_settings', $default_options);
        }
    }
    
    /**
     * عند إلغاء تفعيل الإضافة
     */
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    /**
     * تحميل ملفات الترجمة
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'ai-chat-widget',
            false,
            dirname(AICW_PLUGIN_BASENAME) . '/languages/'
        );
    }
    
    /**
     * تحميل ملفات CSS و JS للواجهة الأمامية
     */
    public function enqueue_frontend_assets() {
        // التحقق من إمكانية عرض النافذة في الصفحة الحالية
        if (!$this->should_display_widget()) {
            return;
        }
        
        // CSS
        wp_enqueue_style(
            'aicw-chat-widget',
            AICW_PLUGIN_URL . 'assets/css/chat-widget.css',
            array(),
            AICW_VERSION
        );
        
        // JS
        wp_enqueue_script(
            'aicw-chat-widget',
            AICW_PLUGIN_URL . 'assets/js/chat-widget.js',
            array('jquery'),
            AICW_VERSION,
            true
        );
        
        // تمرير البيانات للجافاسكربت
        $settings = get_option('aicw_settings', array());
        
        wp_localize_script('aicw-chat-widget', 'aicwConfig', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aicw_chat_nonce'),
            'primaryColor' => $settings['primary_color'] ?? '#0073aa',
            'secondaryColor' => $settings['secondary_color'] ?? '#ffffff',
            'position' => $settings['widget_position'] ?? 'right',
            'welcomeMessage' => $settings['welcome_message'] ?? 'مرحباً! كيف يمكنني مساعدتك؟',
            'botName' => $settings['bot_name'] ?? 'المساعد الذكي',
            'botAvatar' => $settings['bot_avatar'] ?? '',
            'enableSound' => $settings['enable_sound'] ?? true,
            'isRTL' => is_rtl(),
            'strings' => array(
                'placeholder' => __('اكتب رسالتك هنا...', 'ai-chat-widget'),
                'send' => __('إرسال', 'ai-chat-widget'),
                'typing' => __('جاري الكتابة...', 'ai-chat-widget'),
                'error' => __('حدث خطأ. يرجى المحاولة مرة أخرى.', 'ai-chat-widget'),
                'offline' => __('عذراً، الخدمة غير متوفرة حالياً.', 'ai-chat-widget'),
            ),
        ));
    }
    
    /**
     * تحميل ملفات CSS و JS للوحة التحكم
     */
    public function enqueue_admin_assets($hook) {
        // فقط في صفحة إعدادات الإضافة
        if ('toplevel_page_aicw-settings' !== $hook) {
            return;
        }
        
        // CSS
        wp_enqueue_style(
            'aicw-admin-style',
            AICW_PLUGIN_URL . 'assets/css/admin-style.css',
            array(),
            AICW_VERSION
        );
        
        // Media uploader
        wp_enqueue_media();
        
        // JS
        wp_enqueue_script(
            'aicw-admin-script',
            AICW_PLUGIN_URL . 'assets/js/admin-script.js',
            array('jquery', 'wp-color-picker'),
            AICW_VERSION,
            true
        );
        
        // Color picker CSS
        wp_enqueue_style('wp-color-picker');
        
        // تمرير البيانات للجافاسكربت
        wp_localize_script('aicw-admin-script', 'aicwAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aicw_admin_nonce'),
            'strings' => array(
                'selectImage' => __('اختر صورة', 'ai-chat-widget'),
                'useImage' => __('استخدم هذه الصورة', 'ai-chat-widget'),
                'saved' => __('تم الحفظ بنجاح', 'ai-chat-widget'),
                'error' => __('حدث خطأ', 'ai-chat-widget'),
                'testSuccess' => __('تم الاتصال بنجاح!', 'ai-chat-widget'),
                'testFailed' => __('فشل الاتصال. تحقق من مفتاح API.', 'ai-chat-widget'),
            ),
        ));
    }
    
    /**
     * التحقق من إمكانية عرض النافذة
     */
    private function should_display_widget() {
        $settings = get_option('aicw_settings', array());
        
        // التحقق من وجود API Key
        $provider = $settings['ai_provider'] ?? 'openai';
        $api_key = $settings[$provider . '_api_key'] ?? '';
        
        if (empty($api_key)) {
            return false;
        }
        
        // التحقق من الصفحات المستبعدة
        $excluded = $settings['excluded_pages'] ?? '';
        if (!empty($excluded)) {
            $excluded_ids = array_map('trim', explode(',', $excluded));
            if (in_array(get_the_ID(), $excluded_ids)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * عرض نافذة الدردشة
     */
    public function render_chat_widget() {
        if (!$this->should_display_widget()) {
            return;
        }
        
        include AICW_PLUGIN_DIR . 'templates/chat-window.php';
    }
}

/**
 * تشغيل الإضافة
 */
function aicw_init() {
    return AI_Chat_Widget::get_instance();
}

// بدء التشغيل
aicw_init();
