/**
 * AI Chat Widget - Admin JavaScript
 * 
 * @package AI_Chat_Widget
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * Admin Settings Handler
     */
    const AICWAdmin = {
        /**
         * Initialize
         */
        init: function() {
            this.initTabs();
            this.initColorPickers();
            this.initProviderSwitch();
            this.initPasswordToggle();
            this.initApiTest();
            this.initAvatarUpload();
            this.initPromptTemplate();
            this.initRangeSlider();
            this.initExportCSV();
        },

        /**
         * Tab navigation
         */
        initTabs: function() {
            $('.aicw-tab-link').on('click', function(e) {
                e.preventDefault();
                
                const target = $(this).attr('href');
                
                // Update nav
                $('.aicw-tab-link').removeClass('active');
                $(this).addClass('active');
                
                // Update content
                $('.aicw-tab-content').removeClass('active');
                $(target).addClass('active');
                
                // Save to localStorage
                localStorage.setItem('aicw_active_tab', target);
            });
            
            // Restore last active tab
            const savedTab = localStorage.getItem('aicw_active_tab');
            if (savedTab && $(savedTab).length) {
                $('a[href="' + savedTab + '"]').click();
            }
        },

        /**
         * Color pickers
         */
        initColorPickers: function() {
            if ($.fn.wpColorPicker) {
                $('.aicw-color-picker').wpColorPicker();
            }
        },

        /**
         * Provider switch
         */
        initProviderSwitch: function() {
            $('#ai_provider').on('change', function() {
                const provider = $(this).val();
                
                $('.aicw-provider-settings').hide();
                $('#' + provider + '-settings').show();
            });
        },

        /**
         * Password toggle
         */
        initPasswordToggle: function() {
            $('.aicw-toggle-password').on('click', function() {
                const $input = $(this).prev('input');
                const type = $input.attr('type');
                
                if (type === 'password') {
                    $input.attr('type', 'text');
                    $(this).text(aicwAdmin.strings.hide || 'إخفاء');
                } else {
                    $input.attr('type', 'password');
                    $(this).text(aicwAdmin.strings.show || 'إظهار');
                }
            });
        },

        /**
         * API test
         */
        initApiTest: function() {
            $('.aicw-test-api').on('click', function() {
                const $btn = $(this);
                const provider = $btn.data('provider');
                const $apiKeyInput = $('#' + provider + '_api_key');
                const apiKey = $apiKeyInput.val();
                
                if (!apiKey) {
                    alert('يرجى إدخال مفتاح API أولاً');
                    return;
                }
                
                // Remove previous result
                $btn.siblings('.aicw-test-result').remove();
                
                // Show loading
                $btn.prop('disabled', true).text('جاري الاختبار...');
                
                $.ajax({
                    url: aicwAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'aicw_test_api',
                        nonce: aicwAdmin.nonce,
                        provider: provider,
                        api_key: apiKey
                    },
                    success: function(response) {
                        $btn.prop('disabled', false).text('اختبار الاتصال');
                        
                        const resultClass = response.success ? 'success' : 'error';
                        const message = response.data.message;
                        
                        $btn.after('<span class="aicw-test-result ' + resultClass + '">' + message + '</span>');
                        
                        // Remove result after 5 seconds
                        setTimeout(function() {
                            $('.aicw-test-result').fadeOut(function() {
                                $(this).remove();
                            });
                        }, 5000);
                    },
                    error: function() {
                        $btn.prop('disabled', false).text('اختبار الاتصال');
                        $btn.after('<span class="aicw-test-result error">فشل الاتصال</span>');
                    }
                });
            });
        },

        /**
         * Avatar upload
         */
        initAvatarUpload: function() {
            let mediaUploader;
            
            $('.aicw-upload-avatar').on('click', function(e) {
                e.preventDefault();
                
                const $preview = $(this).siblings('.aicw-avatar-preview');
                const $input = $(this).siblings('#bot_avatar');
                const $removeBtn = $(this).siblings('.aicw-remove-avatar');
                
                if (mediaUploader) {
                    mediaUploader.open();
                    return;
                }
                
                mediaUploader = wp.media({
                    title: aicwAdmin.strings.selectImage || 'اختر صورة',
                    button: {
                        text: aicwAdmin.strings.useImage || 'استخدم هذه الصورة'
                    },
                    multiple: false
                });
                
                mediaUploader.on('select', function() {
                    const attachment = mediaUploader.state().get('selection').first().toJSON();
                    $input.val(attachment.url);
                    $preview.html('<img src="' + attachment.url + '" alt="">');
                    $removeBtn.show();
                });
                
                mediaUploader.open();
            });
            
            $('.aicw-remove-avatar').on('click', function() {
                const $preview = $(this).siblings('.aicw-avatar-preview');
                const $input = $(this).siblings('#bot_avatar');
                
                $input.val('');
                $preview.html('');
                $(this).hide();
            });
        },

        /**
         * Prompt template
         */
        initPromptTemplate: function() {
            $('#apply_template').on('click', function() {
                const template = $('#prompt_template').val();
                
                if (template) {
                    $('#system_prompt').val(template);
                }
            });
        },

        /**
         * Range slider value display
         */
        initRangeSlider: function() {
            $('.aicw-range').on('input', function() {
                $(this).next('.aicw-range-value').text($(this).val());
            });
        },

        /**
         * Export CSV
         */
        initExportCSV: function() {
            $('#export-csv').on('click', function() {
                const $btn = $(this);
                
                $btn.prop('disabled', true).text('جاري التصدير...');
                
                $.ajax({
                    url: aicwAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'aicw_export_logs',
                        nonce: aicwAdmin.nonce
                    },
                    success: function(response) {
                        $btn.prop('disabled', false).text('تصدير CSV');
                        
                        if (response.success) {
                            // Create download
                            const blob = new Blob([response.data.csv], { type: 'text/csv;charset=utf-8;' });
                            const link = document.createElement('a');
                            const url = URL.createObjectURL(blob);
                            
                            link.setAttribute('href', url);
                            link.setAttribute('download', 'chat_logs_' + new Date().toISOString().slice(0, 10) + '.csv');
                            link.style.visibility = 'hidden';
                            
                            document.body.appendChild(link);
                            link.click();
                            document.body.removeChild(link);
                        } else {
                            alert(response.data.message || 'فشل التصدير');
                        }
                    },
                    error: function() {
                        $btn.prop('disabled', false).text('تصدير CSV');
                        alert('حدث خطأ أثناء التصدير');
                    }
                });
            });
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        AICWAdmin.init();
    });

})(jQuery);
