/**
 * AI Chat Widget - Frontend JavaScript
 * 
 * @package AI_Chat_Widget
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * AI Chat Widget Class
     */
    class AIChatWidget {
        constructor(config) {
            this.config = config;
            this.messages = [];
            this.isOpen = false;
            this.isTyping = false;
            this.sessionId = this.getSessionId();
            this.unreadCount = 0;
            
            this.init();
        }

        /**
         * Initialize the widget
         */
        init() {
            this.cacheElements();
            this.bindEvents();
            this.loadSavedMessages();
            this.showWelcomeMessage();
        }

        /**
         * Cache DOM elements
         */
        cacheElements() {
            this.$widget = $('#aicw-chat-widget');
            this.$toggle = $('#aicw-chat-toggle');
            this.$window = $('#aicw-chat-window');
            this.$messages = $('#aicw-messages');
            this.$form = $('#aicw-chat-form');
            this.$input = $('#aicw-input');
            this.$sendBtn = $('#aicw-send');
            this.$typing = $('#aicw-typing');
            this.$badge = this.$toggle.find('.aicw-unread-badge');
            this.$iconChat = this.$toggle.find('.aicw-icon-chat');
            this.$iconClose = this.$toggle.find('.aicw-icon-close');
        }

        /**
         * Bind events
         */
        bindEvents() {
            // Toggle chat window
            this.$toggle.on('click', () => this.toggleChat());
            
            // Close button
            this.$widget.find('.aicw-btn-close, .aicw-btn-minimize').on('click', () => this.closeChat());
            
            // Form submit
            this.$form.on('submit', (e) => {
                e.preventDefault();
                this.sendMessage();
            });
            
            // Input handling
            this.$input.on('input', () => this.handleInput());
            this.$input.on('keydown', (e) => this.handleKeydown(e));
            
            // Auto-resize textarea
            this.$input.on('input', () => this.autoResize());
            
            // Click outside to close (optional)
            $(document).on('click', (e) => {
                if (this.isOpen && !$(e.target).closest('#aicw-chat-widget').length) {
                    // Uncomment to enable click outside to close
                    // this.closeChat();
                }
            });
        }

        /**
         * Toggle chat window
         */
        toggleChat() {
            if (this.isOpen) {
                this.closeChat();
            } else {
                this.openChat();
            }
        }

        /**
         * Open chat window
         */
        openChat() {
            this.isOpen = true;
            this.$widget.addClass('aicw-open');
            this.$window.slideDown(300);
            this.$input.focus();
            this.clearUnread();
            this.scrollToBottom();
        }

        /**
         * Close chat window
         */
        closeChat() {
            this.isOpen = false;
            this.$widget.removeClass('aicw-open');
            this.$window.slideUp(300);
        }

        /**
         * Handle input changes
         */
        handleInput() {
            const value = this.$input.val().trim();
            this.$sendBtn.prop('disabled', value.length === 0);
        }

        /**
         * Handle keydown events
         */
        handleKeydown(e) {
            // Send on Enter (without Shift)
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.sendMessage();
            }
        }

        /**
         * Auto-resize textarea
         */
        autoResize() {
            this.$input.css('height', 'auto');
            this.$input.css('height', Math.min(this.$input[0].scrollHeight, 100) + 'px');
        }

        /**
         * Send message
         */
        sendMessage() {
            const message = this.$input.val().trim();
            
            if (!message || this.isTyping) {
                return;
            }

            const history = this.getConversationHistory();
            
            // Add user message
            this.addMessage(message, 'user');
            
            // Clear input
            this.$input.val('').trigger('input');
            this.autoResize();
            
            // Show typing indicator
            this.showTyping();
            
            // Send to server
            this.sendToServer(message, history);
        }

        /**
         * Send message to server
         */
        sendToServer(message, history = []) {
            $.ajax({
                url: this.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'aicw_send_message',
                    nonce: this.config.nonce,
                    message: message,
                    session_id: this.sessionId,
                    conversation_history: JSON.stringify(history)
                },
                success: (response) => {
                    this.hideTyping();
                    
                    if (response.success) {
                        this.addMessage(response.data.message, 'bot');
                        this.playSound();
                    } else {
                        this.addMessage(response.data.message || this.config.strings.error, 'bot', true);
                    }
                },
                error: () => {
                    this.hideTyping();
                    this.addMessage(this.config.strings.error, 'bot', true);
                }
            });
        }

        /**
         * Add message to chat
         */
        addMessage(text, type, isError = false) {
            const time = this.formatTime(new Date());
            const avatar = type === 'bot' ? this.getBotAvatar() : this.getUserAvatar();
            
            const messageHtml = `
                <div class="aicw-message aicw-message-${type} ${isError ? 'aicw-message-error' : ''}">
                    <div class="aicw-message-avatar">${avatar}</div>
                    <div class="aicw-message-content">
                        <div class="aicw-message-bubble">${this.formatMessage(text)}</div>
                        <span class="aicw-message-time">${time}</span>
                    </div>
                </div>
            `;
            
            this.$messages.append(messageHtml);
            this.scrollToBottom();
            
            // Save to storage
            this.messages.push({ text, type, time, isError });
            this.saveMessages();
            
            // Update unread if window is closed
            if (!this.isOpen && type === 'bot') {
                this.incrementUnread();
            }
        }

        /**
         * Get bot avatar HTML
         */
        getBotAvatar() {
            if (this.config.botAvatar) {
                return `<img src="${this.config.botAvatar}" alt="">`;
            }
            return `<svg viewBox="0 0 24 24" fill="currentColor" width="20" height="20">
                <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 3c1.66 0 3 1.34 3 3s-1.34 3-3 3-3-1.34-3-3 1.34-3 3-3zm0 14.2c-2.5 0-4.71-1.28-6-3.22.03-1.99 4-3.08 6-3.08 1.99 0 5.97 1.09 6 3.08-1.29 1.94-3.5 3.22-6 3.22z"/>
            </svg>`;
        }

        /**
         * Get user avatar HTML
         */
        getUserAvatar() {
            return `<svg viewBox="0 0 24 24" fill="currentColor" width="20" height="20">
                <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 3c1.66 0 3 1.34 3 3s-1.34 3-3 3-3-1.34-3-3 1.34-3 3-3zm0 14.2c-2.5 0-4.71-1.28-6-3.22.03-1.99 4-3.08 6-3.08 1.99 0 5.97 1.09 6 3.08-1.29 1.94-3.5 3.22-6 3.22z"/>
            </svg>`;
        }

        /**
         * Show typing indicator
         */
        showTyping() {
            this.isTyping = true;
            this.$typing.slideDown(200);
            this.$sendBtn.prop('disabled', true);
            this.scrollToBottom();
        }

        /**
         * Hide typing indicator
         */
        hideTyping() {
            this.isTyping = false;
            this.$typing.slideUp(200);
            this.$sendBtn.prop('disabled', this.$input.val().trim().length === 0);
        }

        /**
         * Show welcome message
         */
        showWelcomeMessage() {
            if (this.messages.length === 0 && this.config.welcomeMessage) {
                setTimeout(() => {
                    this.addMessage(this.config.welcomeMessage, 'bot');
                }, 500);
            }
        }

        /**
         * Scroll messages to bottom
         */
        scrollToBottom() {
            setTimeout(() => {
                this.$messages.scrollTop(this.$messages[0].scrollHeight);
            }, 100);
        }

        /**
         * Play notification sound
         */
        playSound() {
            if (!this.config.enableSound) {
                return;
            }
            
            try {
                const audio = new Audio('data:audio/mp3;base64,SUQzBAAAAAAAI1RTU0UAAAAPAAADTGF2ZjU4Ljc2LjEwMAAAAAAAAAAAAAAA//tQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAWGluZwAAAA8AAAACAAABhgC7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7v////////////////////////////////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA//tQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAWGluZwAAAA8AAAACAAABhgC7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7u7v////////////////////////////////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA');
                audio.volume = 0.3;
                audio.play().catch(() => {});
            } catch (e) {
                // Ignore audio errors
            }
        }

        /**
         * Format time
         */
        formatTime(date) {
            return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        }

        /**
         * Escape HTML
         */
        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        /**
         * Format message with clickable links
         */
        formatMessage(text) {
            // First escape HTML for security
            let escaped = this.escapeHtml(text);
            
            // URL regex pattern
            const urlPattern = /(https?:\/\/[^\s<>"{}|\\^`\[\]]+)/gi;
            
            // Replace URLs with clickable links
            escaped = escaped.replace(urlPattern, function(url) {
                // Clean up any trailing punctuation that might have been caught
                let cleanUrl = url;
                let trailing = '';
                const trailingPunctuation = /([.,;:!?)]+)$/;
                const match = cleanUrl.match(trailingPunctuation);
                if (match) {
                    trailing = match[1];
                    cleanUrl = cleanUrl.slice(0, -trailing.length);
                }
                return '<a href="' + cleanUrl + '" target="_blank" rel="noopener noreferrer">' + cleanUrl + '</a>' + trailing;
            });
            
            // Email pattern
            const emailPattern = /([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/gi;
            escaped = escaped.replace(emailPattern, '<a href="mailto:$1">$1</a>');
            
            // Convert newlines to breaks
            escaped = escaped.replace(/\n/g, '<br>');
            
            return escaped;
        }

        /**
         * Get or create session ID
         */
        getSessionId() {
            let sessionId = localStorage.getItem('aicw_session_id');
            
            if (!sessionId) {
                sessionId = 'sess_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
                localStorage.setItem('aicw_session_id', sessionId);
            }
            
            return sessionId;
        }

        /**
         * Save messages to localStorage
         */
        saveMessages() {
            try {
                // Keep only last 50 messages
                const toSave = this.messages.slice(-50);
                localStorage.setItem('aicw_messages_' + this.sessionId, JSON.stringify(toSave));
            } catch (e) {
                // Storage full or unavailable
            }
        }

        /**
         * Load saved messages
         */
        loadSavedMessages() {
            try {
                const saved = localStorage.getItem('aicw_messages_' + this.sessionId);
                
                if (saved) {
                    const messages = JSON.parse(saved);
                    
                    // Check if messages are from today
                    const today = new Date().toDateString();
                    const firstMessageDate = messages[0] ? new Date().toDateString() : null;
                    
                    if (firstMessageDate === today) {
                        messages.forEach(msg => {
                            const messageHtml = `
                                <div class="aicw-message aicw-message-${msg.type} ${msg.isError ? 'aicw-message-error' : ''}">
                                    <div class="aicw-message-avatar">${msg.type === 'bot' ? this.getBotAvatar() : this.getUserAvatar()}</div>
                                    <div class="aicw-message-content">
                                        <div class="aicw-message-bubble">${this.formatMessage(msg.text)}</div>
                                        <span class="aicw-message-time">${msg.time}</span>
                                    </div>
                                </div>
                            `;
                            this.$messages.append(messageHtml);
                        });
                        this.messages = messages;
                    } else {
                        // Clear old messages
                        localStorage.removeItem('aicw_messages_' + this.sessionId);
                    }
                }
            } catch (e) {
                // Invalid data
            }
        }

        /**
         * Increment unread count
         */
        incrementUnread() {
            this.unreadCount++;
            this.$badge.text(this.unreadCount).show();
        }

        /**
         * Clear unread count
         */
        clearUnread() {
            this.unreadCount = 0;
            this.$badge.hide();
        }

        /**
         * Get recent conversation history for context
         */
        getConversationHistory(limit = 10) {
            if (!Array.isArray(this.messages) || this.messages.length === 0) {
                return [];
            }

            const filtered = this.messages.filter(msg => !msg.isError);
            const recent = filtered.slice(-limit);

            return recent.map(msg => ({
                role: msg.type === 'bot' ? 'assistant' : 'user',
                content: msg.text || ''
            })).filter(entry => entry.content.trim().length > 0);
        }
    }

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        if (typeof aicwConfig !== 'undefined') {
            window.aicwWidget = new AIChatWidget(aicwConfig);
        }
    });

})(jQuery);
