<?php
/**
 * كلاس صفحة الإعدادات
 * 
 * @package AI_Chat_Widget
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * كلاس إعدادات لوحة التحكم
 */
class AICW_Admin_Settings {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_aicw_test_api', array($this, 'test_api_connection'));
        add_action('wp_ajax_aicw_export_logs', array($this, 'export_logs'));
    }
    
    /**
     * إضافة القائمة في لوحة التحكم
     */
    public function add_admin_menu() {
        add_menu_page(
            __('AI Chat Widget', 'ai-chat-widget'),
            __('AI Chat Widget', 'ai-chat-widget'),
            'manage_options',
            'aicw-settings',
            array($this, 'render_settings_page'),
            'dashicons-format-chat',
            30
        );
        
        add_submenu_page(
            'aicw-settings',
            __('الإعدادات', 'ai-chat-widget'),
            __('الإعدادات', 'ai-chat-widget'),
            'manage_options',
            'aicw-settings'
        );
        
        add_submenu_page(
            'aicw-settings',
            __('سجل المحادثات', 'ai-chat-widget'),
            __('سجل المحادثات', 'ai-chat-widget'),
            'manage_options',
            'aicw-logs',
            array($this, 'render_logs_page')
        );
    }
    
    /**
     * تسجيل الإعدادات
     */
    public function register_settings() {
        register_setting('aicw_settings_group', 'aicw_settings', array(
            'sanitize_callback' => array($this, 'sanitize_settings'),
        ));
    }
    
    /**
     * تنظيف الإعدادات
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        
        // AI Provider
        $sanitized['ai_provider'] = sanitize_text_field($input['ai_provider'] ?? 'openai');
        
        // OpenAI
        $sanitized['openai_api_key'] = sanitize_text_field($input['openai_api_key'] ?? '');
        $sanitized['openai_model'] = sanitize_text_field($input['openai_model'] ?? 'gpt-3.5-turbo');
        
        // DeepSeek
        $sanitized['deepseek_api_key'] = sanitize_text_field($input['deepseek_api_key'] ?? '');
        $sanitized['deepseek_model'] = sanitize_text_field($input['deepseek_model'] ?? 'deepseek-chat');
        
        // AI Settings
        $sanitized['temperature'] = floatval($input['temperature'] ?? 0.7);
        $sanitized['temperature'] = max(0, min(1, $sanitized['temperature']));
        
        $sanitized['max_tokens'] = intval($input['max_tokens'] ?? 1000);
        $sanitized['max_tokens'] = max(100, min(4000, $sanitized['max_tokens']));
        
        // Knowledge Base
        $sanitized['knowledge_base'] = wp_kses_post($input['knowledge_base'] ?? '');
        $sanitized['system_prompt'] = sanitize_textarea_field($input['system_prompt'] ?? '');
        
        // Appearance
        $sanitized['primary_color'] = sanitize_hex_color($input['primary_color'] ?? '#0073aa');
        $sanitized['secondary_color'] = sanitize_hex_color($input['secondary_color'] ?? '#ffffff');
        $sanitized['widget_position'] = sanitize_text_field($input['widget_position'] ?? 'right');
        $sanitized['welcome_message'] = sanitize_textarea_field($input['welcome_message'] ?? '');
        $sanitized['bot_name'] = sanitize_text_field($input['bot_name'] ?? 'المساعد الذكي');
        $sanitized['bot_avatar'] = esc_url_raw($input['bot_avatar'] ?? '');
        
        // Pages
        $sanitized['enabled_pages'] = sanitize_text_field($input['enabled_pages'] ?? 'all');
        $sanitized['excluded_pages'] = sanitize_text_field($input['excluded_pages'] ?? '');
        
        // Other settings
        $sanitized['rate_limit'] = intval($input['rate_limit'] ?? 10);
        $sanitized['enable_sound'] = !empty($input['enable_sound']);
        $sanitized['enable_logging'] = !empty($input['enable_logging']);
        
        return $sanitized;
    }
    
    /**
     * عرض صفحة الإعدادات
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $settings = get_option('aicw_settings', array());
        $providers = AICW_AI_Provider_Factory::get_providers();
        $templates = AICW_Knowledge_Base::get_prompt_templates();
        ?>
        <div class="wrap aicw-admin-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <?php settings_errors('aicw_settings'); ?>
            
            <form method="post" action="options.php" id="aicw-settings-form">
                <?php settings_fields('aicw_settings_group'); ?>
                
                <div class="aicw-tabs">
                    <nav class="aicw-tabs-nav">
                        <a href="#tab-ai" class="aicw-tab-link active"><?php _e('مزود الذكاء الاصطناعي', 'ai-chat-widget'); ?></a>
                        <a href="#tab-knowledge" class="aicw-tab-link"><?php _e('قاعدة المعرفة', 'ai-chat-widget'); ?></a>
                        <a href="#tab-prompt" class="aicw-tab-link"><?php _e('تعليمات البوت', 'ai-chat-widget'); ?></a>
                        <a href="#tab-appearance" class="aicw-tab-link"><?php _e('المظهر', 'ai-chat-widget'); ?></a>
                        <a href="#tab-advanced" class="aicw-tab-link"><?php _e('إعدادات متقدمة', 'ai-chat-widget'); ?></a>
                    </nav>
                    
                    <!-- تبويب مزود AI -->
                    <div id="tab-ai" class="aicw-tab-content active">
                        <h2><?php _e('إعدادات مزود الذكاء الاصطناعي', 'ai-chat-widget'); ?></h2>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="ai_provider"><?php _e('اختر المزود', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <select name="aicw_settings[ai_provider]" id="ai_provider" class="regular-text">
                                        <?php foreach ($providers as $key => $provider) : ?>
                                            <option value="<?php echo esc_attr($key); ?>" <?php selected($settings['ai_provider'] ?? 'openai', $key); ?>>
                                                <?php echo esc_html($provider['name']); ?> - <?php echo esc_html($provider['description']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </td>
                            </tr>
                        </table>
                        
                        <!-- OpenAI Settings -->
                        <div class="aicw-provider-settings" id="openai-settings" style="<?php echo ($settings['ai_provider'] ?? 'openai') !== 'openai' ? 'display:none;' : ''; ?>">
                            <h3><?php _e('إعدادات OpenAI', 'ai-chat-widget'); ?></h3>
                            <table class="form-table">
                                <tr>
                                    <th scope="row">
                                        <label for="openai_api_key"><?php _e('مفتاح API', 'ai-chat-widget'); ?></label>
                                    </th>
                                    <td>
                                        <input type="password" name="aicw_settings[openai_api_key]" id="openai_api_key" 
                                               value="<?php echo esc_attr($settings['openai_api_key'] ?? ''); ?>" class="regular-text">
                                        <button type="button" class="button aicw-toggle-password"><?php _e('إظهار', 'ai-chat-widget'); ?></button>
                                        <button type="button" class="button aicw-test-api" data-provider="openai"><?php _e('اختبار الاتصال', 'ai-chat-widget'); ?></button>
                                        <p class="description">
                                            <?php _e('احصل على مفتاح API من', 'ai-chat-widget'); ?>
                                            <a href="https://platform.openai.com/api-keys" target="_blank">OpenAI Dashboard</a>
                                        </p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row">
                                        <label for="openai_model"><?php _e('النموذج', 'ai-chat-widget'); ?></label>
                                    </th>
                                    <td>
                                        <select name="aicw_settings[openai_model]" id="openai_model" class="regular-text">
                                            <?php foreach ($providers['openai']['models'] as $key => $name) : ?>
                                                <option value="<?php echo esc_attr($key); ?>" <?php selected($settings['openai_model'] ?? 'gpt-3.5-turbo', $key); ?>>
                                                    <?php echo esc_html($name); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                </tr>
                            </table>
                        </div>
                        
                        <!-- DeepSeek Settings -->
                        <div class="aicw-provider-settings" id="deepseek-settings" style="<?php echo ($settings['ai_provider'] ?? 'openai') !== 'deepseek' ? 'display:none;' : ''; ?>">
                            <h3><?php _e('إعدادات DeepSeek', 'ai-chat-widget'); ?></h3>
                            <table class="form-table">
                                <tr>
                                    <th scope="row">
                                        <label for="deepseek_api_key"><?php _e('مفتاح API', 'ai-chat-widget'); ?></label>
                                    </th>
                                    <td>
                                        <input type="password" name="aicw_settings[deepseek_api_key]" id="deepseek_api_key" 
                                               value="<?php echo esc_attr($settings['deepseek_api_key'] ?? ''); ?>" class="regular-text">
                                        <button type="button" class="button aicw-toggle-password"><?php _e('إظهار', 'ai-chat-widget'); ?></button>
                                        <button type="button" class="button aicw-test-api" data-provider="deepseek"><?php _e('اختبار الاتصال', 'ai-chat-widget'); ?></button>
                                        <p class="description">
                                            <?php _e('احصل على مفتاح API من', 'ai-chat-widget'); ?>
                                            <a href="https://platform.deepseek.com/" target="_blank">DeepSeek Platform</a>
                                        </p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row">
                                        <label for="deepseek_model"><?php _e('النموذج', 'ai-chat-widget'); ?></label>
                                    </th>
                                    <td>
                                        <select name="aicw_settings[deepseek_model]" id="deepseek_model" class="regular-text">
                                            <?php foreach ($providers['deepseek']['models'] as $key => $name) : ?>
                                                <option value="<?php echo esc_attr($key); ?>" <?php selected($settings['deepseek_model'] ?? 'deepseek-chat', $key); ?>>
                                                    <?php echo esc_html($name); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                </tr>
                            </table>
                        </div>
                        
                        <!-- إعدادات مشتركة -->
                        <h3><?php _e('إعدادات عامة', 'ai-chat-widget'); ?></h3>
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="temperature"><?php _e('Temperature', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="range" name="aicw_settings[temperature]" id="temperature" 
                                           value="<?php echo esc_attr($settings['temperature'] ?? 0.7); ?>" 
                                           min="0" max="1" step="0.1" class="aicw-range">
                                    <span class="aicw-range-value"><?php echo esc_html($settings['temperature'] ?? 0.7); ?></span>
                                    <p class="description"><?php _e('0 = ردود ثابتة، 1 = ردود إبداعية متنوعة', 'ai-chat-widget'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="max_tokens"><?php _e('الحد الأقصى للكلمات', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="number" name="aicw_settings[max_tokens]" id="max_tokens" 
                                           value="<?php echo esc_attr($settings['max_tokens'] ?? 1000); ?>" 
                                           min="100" max="4000" class="small-text">
                                    <p class="description"><?php _e('الحد الأقصى لعدد التوكنات في كل رد (100-4000)', 'ai-chat-widget'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- تبويب قاعدة المعرفة -->
                    <div id="tab-knowledge" class="aicw-tab-content">
                        <h2><?php _e('قاعدة المعرفة', 'ai-chat-widget'); ?></h2>
                        <p class="description"><?php _e('أضف معلومات عن شركتك ومنتجاتك وخدماتك. سيستخدمها البوت للإجابة على أسئلة العملاء.', 'ai-chat-widget'); ?></p>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="knowledge_base"><?php _e('المحتوى', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <?php
                                    wp_editor(
                                        $settings['knowledge_base'] ?? '',
                                        'knowledge_base',
                                        array(
                                            'textarea_name' => 'aicw_settings[knowledge_base]',
                                            'textarea_rows' => 15,
                                            'media_buttons' => false,
                                            'teeny' => true,
                                        )
                                    );
                                    ?>
                                    <p class="description">
                                        <?php _e('المتغيرات المتاحة:', 'ai-chat-widget'); ?>
                                        <code>{site_name}</code>, <code>{site_url}</code>, <code>{current_date}</code>, <code>{admin_email}</code>
                                    </p>
                                </td>
                            </tr>
                        </table>
                        
                        <div class="aicw-example-box">
                            <h4><?php _e('مثال على قاعدة المعرفة:', 'ai-chat-widget'); ?></h4>
                            <pre>
## عن الشركة
نحن {site_name}، شركة متخصصة في...

## ساعات العمل
السبت - الخميس: 9 صباحاً - 6 مساءً
الجمعة: عطلة

## طرق التواصل
البريد الإلكتروني: {admin_email}
الهاتف: +966 XX XXX XXXX

## المنتجات/الخدمات
1. المنتج الأول - السعر: XXX ريال
2. المنتج الثاني - السعر: XXX ريال

## الأسئلة الشائعة
س: كيف أتواصل معكم؟
ج: يمكنك التواصل عبر...
                            </pre>
                        </div>
                    </div>
                    
                    <!-- تبويب تعليمات البوت -->
                    <div id="tab-prompt" class="aicw-tab-content">
                        <h2><?php _e('تعليمات البوت (System Prompt)', 'ai-chat-widget'); ?></h2>
                        <p class="description"><?php _e('حدد شخصية البوت وطريقة تعامله مع العملاء.', 'ai-chat-widget'); ?></p>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label><?php _e('قوالب جاهزة', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <select id="prompt_template" class="regular-text">
                                        <option value=""><?php _e('-- اختر قالباً --', 'ai-chat-widget'); ?></option>
                                        <?php foreach ($templates as $key => $template) : ?>
                                            <option value="<?php echo esc_attr($template['prompt']); ?>">
                                                <?php echo esc_html($template['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <button type="button" class="button" id="apply_template"><?php _e('تطبيق', 'ai-chat-widget'); ?></button>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="system_prompt"><?php _e('التعليمات', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <textarea name="aicw_settings[system_prompt]" id="system_prompt" 
                                              rows="10" class="large-text"><?php echo esc_textarea($settings['system_prompt'] ?? ''); ?></textarea>
                                    <p class="description">
                                        <?php _e('اكتب التعليمات التي تحدد سلوك البوت وشخصيته.', 'ai-chat-widget'); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- تبويب المظهر -->
                    <div id="tab-appearance" class="aicw-tab-content">
                        <h2><?php _e('إعدادات المظهر', 'ai-chat-widget'); ?></h2>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="primary_color"><?php _e('اللون الرئيسي', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="text" name="aicw_settings[primary_color]" id="primary_color" 
                                           value="<?php echo esc_attr($settings['primary_color'] ?? '#0073aa'); ?>" class="aicw-color-picker">
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="secondary_color"><?php _e('اللون الثانوي', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="text" name="aicw_settings[secondary_color]" id="secondary_color" 
                                           value="<?php echo esc_attr($settings['secondary_color'] ?? '#ffffff'); ?>" class="aicw-color-picker">
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="widget_position"><?php _e('موضع النافذة', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <select name="aicw_settings[widget_position]" id="widget_position">
                                        <option value="right" <?php selected($settings['widget_position'] ?? 'right', 'right'); ?>><?php _e('يمين', 'ai-chat-widget'); ?></option>
                                        <option value="left" <?php selected($settings['widget_position'] ?? 'right', 'left'); ?>><?php _e('يسار', 'ai-chat-widget'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="bot_name"><?php _e('اسم البوت', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="text" name="aicw_settings[bot_name]" id="bot_name" 
                                           value="<?php echo esc_attr($settings['bot_name'] ?? 'المساعد الذكي'); ?>" class="regular-text">
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="welcome_message"><?php _e('رسالة الترحيب', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <textarea name="aicw_settings[welcome_message]" id="welcome_message" 
                                              rows="3" class="large-text"><?php echo esc_textarea($settings['welcome_message'] ?? 'مرحباً! كيف يمكنني مساعدتك اليوم؟'); ?></textarea>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="bot_avatar"><?php _e('صورة البوت', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <div class="aicw-avatar-upload">
                                        <input type="hidden" name="aicw_settings[bot_avatar]" id="bot_avatar" 
                                               value="<?php echo esc_url($settings['bot_avatar'] ?? ''); ?>">
                                        <div class="aicw-avatar-preview">
                                            <?php if (!empty($settings['bot_avatar'])) : ?>
                                                <img src="<?php echo esc_url($settings['bot_avatar']); ?>" alt="">
                                            <?php endif; ?>
                                        </div>
                                        <button type="button" class="button aicw-upload-avatar"><?php _e('اختر صورة', 'ai-chat-widget'); ?></button>
                                        <button type="button" class="button aicw-remove-avatar" <?php echo empty($settings['bot_avatar']) ? 'style="display:none;"' : ''; ?>><?php _e('إزالة', 'ai-chat-widget'); ?></button>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- تبويب الإعدادات المتقدمة -->
                    <div id="tab-advanced" class="aicw-tab-content">
                        <h2><?php _e('إعدادات متقدمة', 'ai-chat-widget'); ?></h2>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="excluded_pages"><?php _e('الصفحات المستبعدة', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="text" name="aicw_settings[excluded_pages]" id="excluded_pages" 
                                           value="<?php echo esc_attr($settings['excluded_pages'] ?? ''); ?>" class="regular-text">
                                    <p class="description"><?php _e('أدخل IDs الصفحات مفصولة بفواصل (مثال: 1,5,10)', 'ai-chat-widget'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="rate_limit"><?php _e('حد الرسائل في الدقيقة', 'ai-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input type="number" name="aicw_settings[rate_limit]" id="rate_limit" 
                                           value="<?php echo esc_attr($settings['rate_limit'] ?? 10); ?>" 
                                           min="0" max="100" class="small-text">
                                    <p class="description"><?php _e('0 لتعطيل الحد. يمنع إساءة الاستخدام.', 'ai-chat-widget'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('خيارات إضافية', 'ai-chat-widget'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aicw_settings[enable_sound]" value="1" 
                                               <?php checked(!empty($settings['enable_sound'])); ?>>
                                        <?php _e('تفعيل صوت الإشعارات', 'ai-chat-widget'); ?>
                                    </label>
                                    <br><br>
                                    <label>
                                        <input type="checkbox" name="aicw_settings[enable_logging]" value="1" 
                                               <?php checked(!empty($settings['enable_logging'])); ?>>
                                        <?php _e('تسجيل المحادثات في قاعدة البيانات', 'ai-chat-widget'); ?>
                                    </label>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <?php submit_button(__('حفظ الإعدادات', 'ai-chat-widget')); ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * عرض صفحة سجل المحادثات
     */
    public function render_logs_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $stats = AICW_Chat_Handler::get_stats(30);
        $conversations = AICW_Chat_Handler::get_recent_conversations(50);
        ?>
        <div class="wrap aicw-admin-wrap">
            <h1><?php _e('سجل المحادثات', 'ai-chat-widget'); ?></h1>
            
            <!-- الإحصائيات -->
            <div class="aicw-stats-grid">
                <div class="aicw-stat-box">
                    <span class="aicw-stat-number"><?php echo esc_html($stats['total_messages'] ?? 0); ?></span>
                    <span class="aicw-stat-label"><?php _e('إجمالي الرسائل', 'ai-chat-widget'); ?></span>
                </div>
                <div class="aicw-stat-box">
                    <span class="aicw-stat-number"><?php echo esc_html($stats['unique_sessions'] ?? 0); ?></span>
                    <span class="aicw-stat-label"><?php _e('جلسات فريدة', 'ai-chat-widget'); ?></span>
                </div>
                <div class="aicw-stat-box">
                    <span class="aicw-stat-number"><?php echo number_format($stats['total_tokens'] ?? 0); ?></span>
                    <span class="aicw-stat-label"><?php _e('التوكنات المستخدمة', 'ai-chat-widget'); ?></span>
                </div>
            </div>
            
            <!-- أزرار التصدير -->
            <div class="aicw-actions">
                <button type="button" class="button button-primary" id="export-csv">
                    <?php _e('تصدير CSV', 'ai-chat-widget'); ?>
                </button>
            </div>
            
            <!-- جدول المحادثات -->
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 50px;"><?php _e('ID', 'ai-chat-widget'); ?></th>
                        <th style="width: 120px;"><?php _e('الجلسة', 'ai-chat-widget'); ?></th>
                        <th><?php _e('رسالة المستخدم', 'ai-chat-widget'); ?></th>
                        <th><?php _e('رد البوت', 'ai-chat-widget'); ?></th>
                        <th style="width: 80px;"><?php _e('التوكنات', 'ai-chat-widget'); ?></th>
                        <th style="width: 150px;"><?php _e('التاريخ', 'ai-chat-widget'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($conversations)) : ?>
                        <tr>
                            <td colspan="6"><?php _e('لا توجد محادثات بعد.', 'ai-chat-widget'); ?></td>
                        </tr>
                    <?php else : ?>
                        <?php foreach ($conversations as $conv) : ?>
                            <tr>
                                <td><?php echo esc_html($conv['id']); ?></td>
                                <td><code><?php echo esc_html(substr($conv['session_id'], 0, 10)); ?>...</code></td>
                                <td><?php echo esc_html(wp_trim_words($conv['user_message'], 20)); ?></td>
                                <td><?php echo esc_html(wp_trim_words($conv['bot_response'], 20)); ?></td>
                                <td><?php echo esc_html($conv['tokens_used']); ?></td>
                                <td><?php echo esc_html($conv['created_at']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }
    
    /**
     * اختبار اتصال API
     */
    public function test_api_connection() {
        check_ajax_referer('aicw_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('غير مصرح', 'ai-chat-widget')));
        }
        
        $provider_name = sanitize_text_field($_POST['provider'] ?? 'openai');
        $api_key = sanitize_text_field($_POST['api_key'] ?? '');
        
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('مفتاح API مطلوب', 'ai-chat-widget')));
        }
        
        // إنشاء المزود واختباره
        if ($provider_name === 'deepseek') {
            $provider = new AICW_DeepSeek_Provider($api_key, 'deepseek-chat');
        } else {
            $provider = new AICW_OpenAI_Provider($api_key, 'gpt-3.5-turbo');
        }
        
        $is_valid = $provider->validate_api_key();
        
        if ($is_valid) {
            wp_send_json_success(array('message' => __('تم الاتصال بنجاح!', 'ai-chat-widget')));
        } else {
            wp_send_json_error(array('message' => __('فشل الاتصال. تحقق من مفتاح API.', 'ai-chat-widget')));
        }
    }
    
    /**
     * تصدير السجلات
     */
    public function export_logs() {
        check_ajax_referer('aicw_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('غير مصرح', 'ai-chat-widget')));
        }
        
        $csv = AICW_Chat_Handler::export_to_csv(30);
        
        if ($csv === false) {
            wp_send_json_error(array('message' => __('لا توجد بيانات للتصدير', 'ai-chat-widget')));
        }
        
        wp_send_json_success(array('csv' => $csv));
    }
}

// تشغيل كلاس الإعدادات
new AICW_Admin_Settings();
