<?php
/**
 * كلاس مزودي الذكاء الاصطناعي
 * 
 * @package AI_Chat_Widget
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Abstract class لمزودي AI
 */
abstract class AICW_AI_Provider {
    
    /**
     * مفتاح API
     */
    protected $api_key;
    
    /**
     * النموذج المستخدم
     */
    protected $model;
    
    /**
     * Temperature
     */
    protected $temperature;
    
    /**
     * Max tokens
     */
    protected $max_tokens;
    
    /**
     * Constructor
     */
    public function __construct($api_key, $model, $temperature = 0.7, $max_tokens = 1000) {
        $this->api_key = $api_key;
        $this->model = $model;
        $this->temperature = floatval($temperature);
        $this->max_tokens = intval($max_tokens);
    }
    
    /**
     * إرسال رسالة - يجب تنفيذها في الكلاسات الفرعية
     */
    abstract public function send_message($message, $context = '', $system_prompt = '', $history = array());
    
    /**
     * التحقق من صلاحية API Key
     */
    abstract public function validate_api_key();
    
    /**
     * الحصول على النماذج المتاحة
     */
    abstract public static function get_available_models();
    
    /**
     * تنفيذ طلب HTTP
     */
    protected function make_request($url, $body, $headers = array()) {
        $default_headers = array(
            'Content-Type' => 'application/json',
        );
        
        $headers = array_merge($default_headers, $headers);
        
        $response = wp_remote_post($url, array(
            'timeout' => 60,
            'headers' => $headers,
            'body' => json_encode($body),
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message(),
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code !== 200) {
            $error_message = isset($data['error']['message']) 
                ? $data['error']['message'] 
                : __('خطأ غير معروف من API', 'ai-chat-widget');
            
            return array(
                'success' => false,
                'error' => $error_message,
                'code' => $response_code,
            );
        }
        
        return array(
            'success' => true,
            'data' => $data,
        );
    }
}

/**
 * OpenAI Provider
 */
class AICW_OpenAI_Provider extends AICW_AI_Provider {
    
    /**
     * API Endpoint
     */
    const API_URL = 'https://api.openai.com/v1/chat/completions';
    
    /**
     * إرسال رسالة
     */
    public function send_message($message, $context = '', $system_prompt = '', $history = array()) {
        $messages = array();
        
        // System prompt
        $full_system_prompt = $system_prompt;
        if (!empty($context)) {
            $full_system_prompt .= "\n\n### قاعدة المعرفة:\n" . $context;
        }
        
        if (!empty($full_system_prompt)) {
            $messages[] = array(
                'role' => 'system',
                'content' => $full_system_prompt,
            );
        }

        if (!empty($history)) {
            foreach ($history as $entry) {
                if (empty($entry['content'])) {
                    continue;
                }
                $messages[] = array(
                    'role' => $entry['role'] === 'assistant' ? 'assistant' : 'user',
                    'content' => $entry['content'],
                );
            }
        }
        
        // رسالة المستخدم
        $messages[] = array(
            'role' => 'user',
            'content' => $message,
        );
        
        $body = array(
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => $this->temperature,
            'max_tokens' => $this->max_tokens,
        );
        
        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_key,
        );
        
        $response = $this->make_request(self::API_URL, $body, $headers);
        
        if (!$response['success']) {
            return $response;
        }
        
        $data = $response['data'];
        
        return array(
            'success' => true,
            'message' => $data['choices'][0]['message']['content'] ?? '',
            'tokens_used' => $data['usage']['total_tokens'] ?? 0,
        );
    }
    
    /**
     * التحقق من صلاحية API Key
     */
    public function validate_api_key() {
        $body = array(
            'model' => 'gpt-3.5-turbo',
            'messages' => array(
                array('role' => 'user', 'content' => 'Hi'),
            ),
            'max_tokens' => 5,
        );
        
        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_key,
        );
        
        $response = $this->make_request(self::API_URL, $body, $headers);
        
        return $response['success'];
    }
    
    /**
     * الحصول على النماذج المتاحة
     */
    public static function get_available_models() {
        return array(
            'gpt-4o' => 'GPT-4o (الأحدث)',
            'gpt-4o-mini' => 'GPT-4o Mini (سريع وموفر)',
            'gpt-4-turbo' => 'GPT-4 Turbo',
            'gpt-4' => 'GPT-4',
            'gpt-3.5-turbo' => 'GPT-3.5 Turbo (اقتصادي)',
        );
    }
}

/**
 * DeepSeek Provider
 */
class AICW_DeepSeek_Provider extends AICW_AI_Provider {
    
    /**
     * API Endpoint
     */
    const API_URL = 'https://api.deepseek.com/v1/chat/completions';
    
    /**
     * إرسال رسالة
     */
    public function send_message($message, $context = '', $system_prompt = '', $history = array()) {
        $messages = array();
        
        // System prompt
        $full_system_prompt = $system_prompt;
        if (!empty($context)) {
            $full_system_prompt .= "\n\n### قاعدة المعرفة:\n" . $context;
        }
        
        if (!empty($full_system_prompt)) {
            $messages[] = array(
                'role' => 'system',
                'content' => $full_system_prompt,
            );
        }

        if (!empty($history)) {
            foreach ($history as $entry) {
                if (empty($entry['content'])) {
                    continue;
                }
                $messages[] = array(
                    'role' => $entry['role'] === 'assistant' ? 'assistant' : 'user',
                    'content' => $entry['content'],
                );
            }
        }
        
        // رسالة المستخدم
        $messages[] = array(
            'role' => 'user',
            'content' => $message,
        );
        
        $body = array(
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => $this->temperature,
            'max_tokens' => $this->max_tokens,
        );
        
        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_key,
        );
        
        $response = $this->make_request(self::API_URL, $body, $headers);
        
        if (!$response['success']) {
            return $response;
        }
        
        $data = $response['data'];
        
        return array(
            'success' => true,
            'message' => $data['choices'][0]['message']['content'] ?? '',
            'tokens_used' => $data['usage']['total_tokens'] ?? 0,
        );
    }
    
    /**
     * التحقق من صلاحية API Key
     */
    public function validate_api_key() {
        $body = array(
            'model' => 'deepseek-chat',
            'messages' => array(
                array('role' => 'user', 'content' => 'Hi'),
            ),
            'max_tokens' => 5,
        );
        
        $headers = array(
            'Authorization' => 'Bearer ' . $this->api_key,
        );
        
        $response = $this->make_request(self::API_URL, $body, $headers);
        
        return $response['success'];
    }
    
    /**
     * الحصول على النماذج المتاحة
     */
    public static function get_available_models() {
        return array(
            'deepseek-chat' => 'DeepSeek Chat (محادثة عامة)',
            'deepseek-coder' => 'DeepSeek Coder (برمجة)',
            'deepseek-reasoner' => 'DeepSeek Reasoner (تفكير متقدم)',
        );
    }
}

/**
 * Factory class لإنشاء مزودي AI
 */
class AICW_AI_Provider_Factory {
    
    /**
     * إنشاء مزود AI
     */
    public static function create($provider_name = null) {
        $settings = get_option('aicw_settings', array());
        
        if ($provider_name === null) {
            $provider_name = $settings['ai_provider'] ?? 'openai';
        }
        
        switch ($provider_name) {
            case 'deepseek':
                return new AICW_DeepSeek_Provider(
                    $settings['deepseek_api_key'] ?? '',
                    $settings['deepseek_model'] ?? 'deepseek-chat',
                    $settings['temperature'] ?? 0.7,
                    $settings['max_tokens'] ?? 1000
                );
            
            case 'openai':
            default:
                return new AICW_OpenAI_Provider(
                    $settings['openai_api_key'] ?? '',
                    $settings['openai_model'] ?? 'gpt-3.5-turbo',
                    $settings['temperature'] ?? 0.7,
                    $settings['max_tokens'] ?? 1000
                );
        }
    }
    
    /**
     * الحصول على قائمة المزودين
     */
    public static function get_providers() {
        return array(
            'openai' => array(
                'name' => 'OpenAI',
                'description' => 'ChatGPT و GPT-4',
                'models' => AICW_OpenAI_Provider::get_available_models(),
            ),
            'deepseek' => array(
                'name' => 'DeepSeek',
                'description' => 'نماذج DeepSeek المتقدمة',
                'models' => AICW_DeepSeek_Provider::get_available_models(),
            ),
        );
    }
}
