<?php
/**
 * كلاس معالجة الدردشة
 * 
 * @package AI_Chat_Widget
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * كلاس معالج الرسائل
 */
class AICW_Chat_Handler {
    
    /**
     * معالجة الرسالة الواردة
     */
    public static function handle_message() {
        // التحقق من nonce
        if (!check_ajax_referer('aicw_chat_nonce', 'nonce', false)) {
            wp_send_json_error(array(
                'message' => __('فشل التحقق الأمني.', 'ai-chat-widget'),
            ), 403);
        }
        
        // الحصول على الرسالة
        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';
        $session_id = isset($_POST['session_id']) ? sanitize_text_field(wp_unslash($_POST['session_id'])) : '';
        $history = self::extract_history($_POST['conversation_history'] ?? '');
        
        if (empty($message)) {
            wp_send_json_error(array(
                'message' => __('الرسالة فارغة.', 'ai-chat-widget'),
            ), 400);
        }
        
        // التحقق من Rate Limiting
        if (!self::check_rate_limit($session_id)) {
            wp_send_json_error(array(
                'message' => __('لقد أرسلت رسائل كثيرة. انتظر قليلاً.', 'ai-chat-widget'),
            ), 429);
        }
        
        // الحصول على الإعدادات
        $settings = get_option('aicw_settings', array());
        
        // إنشاء مزود AI
        $provider = AICW_AI_Provider_Factory::create();
        
        // الحصول على قاعدة المعرفة والتعليمات
        $knowledge_base = AICW_Knowledge_Base::get_content();
        $system_prompt = AICW_Knowledge_Base::get_system_prompt();
        
        // إرسال الرسالة للـ AI
        $response = $provider->send_message($message, $knowledge_base, $system_prompt, $history);
        
        if (!$response['success']) {
            wp_send_json_error(array(
                'message' => $response['error'] ?? __('حدث خطأ في الاتصال.', 'ai-chat-widget'),
            ), 500);
        }
        
        // تسجيل المحادثة
        if (!empty($settings['enable_logging'])) {
            self::log_conversation($session_id, $message, $response['message'], $response['tokens_used'] ?? 0);
        }
        
        wp_send_json_success(array(
            'message' => $response['message'],
            'tokens' => $response['tokens_used'] ?? 0,
        ));
    }
    
    /**
     * التحقق من Rate Limit
     */
    private static function check_rate_limit($session_id) {
        $settings = get_option('aicw_settings', array());
        $rate_limit = intval($settings['rate_limit'] ?? 10);
        
        if ($rate_limit <= 0) {
            return true; // Rate limiting معطل
        }
        
        $transient_key = 'aicw_rate_' . md5($session_id . $_SERVER['REMOTE_ADDR']);
        $count = get_transient($transient_key);
        
        if ($count === false) {
            set_transient($transient_key, 1, MINUTE_IN_SECONDS);
            return true;
        }
        
        if ($count >= $rate_limit) {
            return false;
        }
        
        set_transient($transient_key, $count + 1, MINUTE_IN_SECONDS);
        return true;
    }
    
    /**
     * تسجيل المحادثة في قاعدة البيانات
     */
    private static function log_conversation($session_id, $user_message, $bot_response, $tokens_used) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aicw_chat_logs';
        
        $wpdb->insert(
            $table_name,
            array(
                'session_id' => $session_id,
                'user_message' => $user_message,
                'bot_response' => $bot_response,
                'tokens_used' => $tokens_used,
                'ip_address' => self::get_client_ip(),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? substr($_SERVER['HTTP_USER_AGENT'], 0, 500) : '',
                'created_at' => current_time('mysql'),
            ),
            array('%s', '%s', '%s', '%d', '%s', '%s', '%s')
        );
    }
    
    /**
     * الحصول على IP العميل
     */
    private static function get_client_ip() {
        $ip_keys = array(
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR',
        );
        
        foreach ($ip_keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                // في حالة X-Forwarded-For قد تكون هناك عدة IPs
                if (strpos($ip, ',') !== false) {
                    $ip = explode(',', $ip)[0];
                }
                return sanitize_text_field(trim($ip));
            }
        }
        
        return '0.0.0.0';
    }

    /**
     * استخراج وتنظيف سجل المحادثة القادم من الواجهة
     */
    private static function extract_history($raw_history) {
        if (empty($raw_history)) {
            return array();
        }

        if (is_string($raw_history)) {
            $decoded = json_decode(wp_unslash($raw_history), true);
        } elseif (is_array($raw_history)) {
            $decoded = $raw_history;
        } else {
            $decoded = array();
        }

        if (!is_array($decoded) || empty($decoded)) {
            return array();
        }

        $history = array();

        foreach ($decoded as $item) {
            if (!is_array($item)) {
                continue;
            }

            $role = isset($item['role']) && 'assistant' === $item['role'] ? 'assistant' : 'user';
            $content = isset($item['content']) ? sanitize_textarea_field($item['content']) : '';

            if ('' === $content) {
                continue;
            }

            $history[] = array(
                'role' => $role,
                'content' => $content,
            );
        }

        if (empty($history)) {
            return array();
        }

        return array_slice($history, -10);
    }
    
    /**
     * الحصول على إحصائيات المحادثات
     */
    public static function get_stats($days = 30) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aicw_chat_logs';
        
        $stats = array();
        
        // إجمالي المحادثات
        $stats['total_messages'] = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days
            )
        );
        
        // عدد الجلسات الفريدة
        $stats['unique_sessions'] = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(DISTINCT session_id) FROM $table_name WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days
            )
        );
        
        // إجمالي التوكنات المستخدمة
        $stats['total_tokens'] = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT SUM(tokens_used) FROM $table_name WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days
            )
        ) ?? 0;
        
        // المحادثات اليومية
        $stats['daily'] = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DATE(created_at) as date, COUNT(*) as count 
                FROM $table_name 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
                GROUP BY DATE(created_at)
                ORDER BY date DESC",
                $days
            ),
            ARRAY_A
        );
        
        return $stats;
    }
    
    /**
     * الحصول على آخر المحادثات
     */
    public static function get_recent_conversations($limit = 50) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aicw_chat_logs';
        
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name ORDER BY created_at DESC LIMIT %d",
                $limit
            ),
            ARRAY_A
        );
    }
    
    /**
     * حذف المحادثات القديمة
     */
    public static function cleanup_old_logs($days = 90) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aicw_chat_logs';
        
        return $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $table_name WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days
            )
        );
    }
    
    /**
     * تصدير المحادثات إلى CSV
     */
    public static function export_to_csv($days = 30) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aicw_chat_logs';
        
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY) ORDER BY created_at DESC",
                $days
            ),
            ARRAY_A
        );
        
        if (empty($results)) {
            return false;
        }
        
        $csv = "ID,Session ID,User Message,Bot Response,Tokens Used,IP Address,Created At\n";
        
        foreach ($results as $row) {
            $csv .= sprintf(
                '"%s","%s","%s","%s","%s","%s","%s"' . "\n",
                $row['id'],
                $row['session_id'],
                str_replace('"', '""', $row['user_message']),
                str_replace('"', '""', $row['bot_response']),
                $row['tokens_used'],
                $row['ip_address'],
                $row['created_at']
            );
        }
        
        return $csv;
    }
}
