<?php
/**
 * كلاس قاعدة المعرفة
 * 
 * @package AI_Chat_Widget
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * كلاس إدارة قاعدة المعرفة
 */
class AICW_Knowledge_Base {
    
    /**
     * الحصول على محتوى قاعدة المعرفة
     */
    public static function get_content() {
        $settings = get_option('aicw_settings', array());
        $knowledge_base = $settings['knowledge_base'] ?? '';
        
        // إضافة معلومات ديناميكية
        $knowledge_base = self::replace_variables($knowledge_base);
        
        return $knowledge_base;
    }
    
    /**
     * استبدال المتغيرات الديناميكية
     */
    public static function replace_variables($text) {
        $variables = array(
            '{site_name}' => get_bloginfo('name'),
            '{site_url}' => home_url(),
            '{site_description}' => get_bloginfo('description'),
            '{current_date}' => date_i18n(get_option('date_format')),
            '{current_time}' => date_i18n(get_option('time_format')),
            '{current_year}' => date('Y'),
            '{admin_email}' => get_option('admin_email'),
        );
        
        return str_replace(array_keys($variables), array_values($variables), $text);
    }
    
    /**
     * الحصول على تعليمات النظام
     */
    public static function get_system_prompt() {
        $settings = get_option('aicw_settings', array());
        $system_prompt = $settings['system_prompt'] ?? '';
        
        // إضافة المتغيرات
        $system_prompt = self::replace_variables($system_prompt);
        
        // إضافة تعليمات أساسية
        $base_instructions = self::get_base_instructions();
        
        return $base_instructions . "\n\n" . $system_prompt;
    }
    
    /**
     * التعليمات الأساسية
     */
    private static function get_base_instructions() {
        $site_name = get_bloginfo('name');
        
        return "أنت مساعد ذكي لموقع {$site_name}. 
- أجب بشكل مهذب ومختصر
- إذا لم تعرف الإجابة، قل ذلك بوضوح
- استخدم المعلومات من قاعدة المعرفة المرفقة
- لا تخترع معلومات غير موجودة
- تحدث بنفس لغة السائل";
    }
    
    /**
     * البحث في قاعدة المعرفة
     */
    public static function search($query) {
        $content = self::get_content();
        
        if (empty($content)) {
            return '';
        }
        
        // تقسيم المحتوى إلى فقرات
        $paragraphs = preg_split('/\n\s*\n/', $content);
        $relevant = array();
        
        // البحث عن الفقرات ذات الصلة
        $query_words = preg_split('/\s+/', mb_strtolower($query));
        
        foreach ($paragraphs as $paragraph) {
            $paragraph = trim($paragraph);
            if (empty($paragraph)) {
                continue;
            }
            
            $score = 0;
            $lower_paragraph = mb_strtolower($paragraph);
            
            foreach ($query_words as $word) {
                if (mb_strlen($word) > 2 && mb_strpos($lower_paragraph, $word) !== false) {
                    $score++;
                }
            }
            
            if ($score > 0) {
                $relevant[] = array(
                    'text' => $paragraph,
                    'score' => $score,
                );
            }
        }
        
        // ترتيب حسب الصلة
        usort($relevant, function($a, $b) {
            return $b['score'] - $a['score'];
        });
        
        // إرجاع أفضل 3 نتائج أو كل المحتوى إذا كان قصيراً
        if (mb_strlen($content) < 2000) {
            return $content;
        }
        
        $result = array();
        $count = 0;
        foreach ($relevant as $item) {
            if ($count >= 3) break;
            $result[] = $item['text'];
            $count++;
        }
        
        return implode("\n\n", $result);
    }
    
    /**
     * الحصول على القوالب الجاهزة
     */
    public static function get_prompt_templates() {
        return array(
            'customer_service' => array(
                'name' => __('خدمة العملاء', 'ai-chat-widget'),
                'prompt' => 'أنت موظف خدمة عملاء محترف ومهذب. مهمتك مساعدة العملاء في استفساراتهم وحل مشاكلهم.
- استمع جيداً لمشكلة العميل
- قدم حلولاً واضحة ومحددة
- اعتذر عن أي إزعاج
- إذا لم تستطع حل المشكلة، وجه العميل للتواصل مع فريق الدعم',
            ),
            'sales' => array(
                'name' => __('مبيعات', 'ai-chat-widget'),
                'prompt' => 'أنت مستشار مبيعات ودود ومحترف. ساعد العملاء في اختيار المنتجات المناسبة لهم.
- اسأل عن احتياجات العميل
- قدم توصيات مناسبة
- اشرح مميزات المنتجات
- لا تكن ملحاً أو مزعجاً
- قدم معلومات عن العروض والخصومات إن وجدت',
            ),
            'technical_support' => array(
                'name' => __('دعم فني', 'ai-chat-widget'),
                'prompt' => 'أنت خبير دعم فني. ساعد العملاء في حل المشاكل التقنية.
- اطلب تفاصيل المشكلة
- قدم خطوات واضحة ومرقمة
- تأكد من فهم العميل للحل
- إذا كانت المشكلة معقدة، اطلب معلومات إضافية',
            ),
            'general' => array(
                'name' => __('مساعد عام', 'ai-chat-widget'),
                'prompt' => 'أنت مساعد ذكي ودود. ساعد الزوار في الإجابة على أسئلتهم حول الموقع والخدمات.
- كن مهذباً ومرحباً
- أجب بشكل مختصر ومفيد
- وجه الزوار للصفحات المناسبة',
            ),
        );
    }
    
    /**
     * تحقق من صحة قاعدة المعرفة
     */
    public static function validate($content) {
        if (empty($content)) {
            return array(
                'valid' => true,
                'warnings' => array(__('قاعدة المعرفة فارغة. البوت سيستخدم معرفته العامة فقط.', 'ai-chat-widget')),
            );
        }
        
        $warnings = array();
        $word_count = str_word_count($content);
        
        if ($word_count < 50) {
            $warnings[] = __('قاعدة المعرفة قصيرة جداً. ننصح بإضافة المزيد من المعلومات.', 'ai-chat-widget');
        }
        
        if ($word_count > 10000) {
            $warnings[] = __('قاعدة المعرفة طويلة جداً. قد يؤثر ذلك على دقة الإجابات.', 'ai-chat-widget');
        }
        
        return array(
            'valid' => true,
            'word_count' => $word_count,
            'warnings' => $warnings,
        );
    }
}
